// Amigrator.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"
#include "Link.h"
#include "AreaGroup.h"
#include "Area.h"

bool backupCfg();
void convertLinks();
void convertAreas();

void resolve(const char* data, string& resolvedData);

int main(int argc, char* argv[])
{
	try {
		std::cout << "Amigrator 1.1 migration tool for Adrenalin\n"
				  << "Converts links/areas from Adrenalin v0.69alpha4 to Adrenalin v0.70alpha1\n";

		if (backupCfg()) {
			convertLinks();
			convertAreas();
		} else {
			std::cout << "Failed to backup previous xml configs\n";
		}
	} catch (string& e) {
		std::cout << "Exception occured: " << e.c_str() << std::endl;
	}

	return 0;
}

char	ADRENALIN_CFG[] = "adrenalin.cfg";
char	AREAS_CFG[]		= "areas.cfg";

VLinks	links;
PLink	link;
int		maxLinkPriority = 0;

VAreaGroups	groups;

VAreas	areas;
PArea	area;


bool checkMask(const char* szValue, const char* szMask) {
    while (*szMask != '\0') {
        switch (*szMask) {
        case '?':
            szMask++;
            if (*szValue != '\0')
                szValue++;
            break;
        case '*':
            szMask++;
            while (*szValue != '\0') {
                if (checkMask(szValue, szMask))
                    break;
                else
                    szValue++;
            }
            if (*szValue != '\0' || checkMask(szValue, szMask))
                return (true);
            else
                return (false);
        default:
			// we can avoid this check, but code is more clear
			if ((*szValue) == '\0')
				return (false);
            if (toupper(*szMask) != toupper(*szValue))
                return (false);
            szMask++;
            szValue++;
        }
    }
    return (*szValue == '\0');
}


string getMaskedAddress(const string& address) {
	string	maskedAddress = address;
	int		pos           = -1;
	// check '/'
	pos = maskedAddress.find( '/' );
	if (pos < 0) {
		// there is no '/'
		maskedAddress = string("*/") + address;
	}
	else {
		// check ':'
		pos = maskedAddress.find( ':' );
		if (pos < 0) {
			maskedAddress = string("*:") + address;
		}
	}
	// remove ".0" if present
	pos = maskedAddress.find( string(".0") );
	if (pos >= 0) {
		maskedAddress = maskedAddress.substr( 0, pos ) +
						maskedAddress.substr( pos + 2 );
	}
	// check '@'
	pos = maskedAddress.find( '@' );
	if (pos < 0) {
		maskedAddress = maskedAddress + string("@*");
	}
	return maskedAddress;
}


bool areAddressesEqual(const string& addr1, const string& addr2) {
	return checkMask( getMaskedAddress( addr1 ).c_str(), getMaskedAddress( addr2 ).c_str() );
}


void getPriorities(const string& address, int& sendPriority, int& receivePriority) {
	for (int i = 0; i < links.size(); i++) {
		if (areAddressesEqual( links[i]->address, address )) {
			sendPriority    = links[i]->sendPriority;
			receivePriority = links[i]->receivePriority;
			return;
		}
	}
	throw string("Failed to find link by address: ") + address;
}


bool seeDoubleQuote(const char* line) {
	while (*line != '\0' && (*line == ' ' || *line == '\t')) {
		++line;
	}
	return (*line == '"');
}

const char* getWord(const char* line, char* word) {
	const char*	s = line;
	// skip spaces
	while (*s != '\0' && (*s == ' ' || *s == '\t')) {
		++s;
	}
	const char*	start = s;
	// count letters
	if (*s == '"') {
		++s;
		++start;
		while (*s != '\0' && *s != '"') {
			++s;
		}
		if (s > start) {
			strncpy( word, start, s - start );
			word[s-start] = '\0';
		}
		else {
			word[0] = '\0';
		}
		if (*s == '"') {
			++s;
		}
	} else {
		while (*s != '\0' && *s != ' ' && *s != '\t') {
			++s;
		}
		if (s > start) {
			strncpy( word, start, s - start );
			word[s-start] = '\0';
		}
		else {
			word[0] = '\0';
		}
	}
	return s;
}


string unrollGroups(const string& groups) {
	string	unrolled;
	char	first, last;
	int		i = 0;
	while (i < groups.size()) {
		switch(groups[i]) {
		case '-':
			if (unrolled.size() > 0) {
				first = unrolled[unrolled.size()-1];
			} else {
				first = 'A';
			}
			++i;
			++first;
			if (i < groups.size()) {
				last = toupper(groups[i]);
				if (last > 'Z')
					last = 'Z';
				while (first < last) {
					unrolled.append( 1, first );
					++first;
				}
			}
			break;
		case '*':
			if (unrolled.size() > 0) {
				first = unrolled[unrolled.size()-1];
			} else {
				first = 'A';
			}
			++first;
			while (first <= 'Z') {
				unrolled.append( 1, first );
				++first;
			}
			i = groups.size();
			break;
		default:
			unrolled.append( 1, toupper(groups[i]) );
			++i;
		}
	}
	return unrolled;
}

void processLinkString(const char* s) {
	char		word[4096];
	const char*	nextPart = NULL;

	nextPart = getWord( s, word );
	if (stricmp( word, "link" ) == 0) {
		// new link definition
		// we can be reading link
		if (link != NULL) {
			links.push_back( link );
		}
		link = new CLink();
		// address
		nextPart = getWord( nextPart, word );
		link->address = word;
		// password
		nextPart = getWord( nextPart, word );
		link->password = word;
		while (*nextPart != '\0') {
			nextPart = getWord( nextPart, word );
			if (*word == '\0') {
				break;
			}
			const char*	slash = strstr( word, "/" );
			if (slash != NULL) {
				// priority/groups
				string	priority( word, slash-word );
				int     scanResult = sscanf( priority.c_str(), "%d", &(link->priority) );
				if (scanResult == 0 || scanResult == EOF || link->priority < 0) {
					link->priority = 0;
				}
				
				link->groups = slash + 1;
			} else {
				switch (word[0]) {
				case 'S':
				case 's':
					link->routing = "send";
					break;
				case 'R':
				case 'r':
					link->routing = "receive";
					break;
				case 'B':
				case 'b':
					link->routing = "bidirectional";
					break;
				case 'A':
				case 'a':
					link->allowAreaCreation = true;
					break;
				case 'N':
				case 'n':
					link->flavour = "normal";
					break;
				case 'D':
				case 'd':
					link->flavour = "direct";
					break;
				case 'H':
				case 'h':
					link->flavour = "hold";
					break;
				case 'C':
				case 'c':
					link->flavour = "crash";
					break;
				case 'I':
				case 'i':
					link->flavour = "immediate";
					break;
				case 'P':
				case 'p':
					link->status = "passive";
					break;
				case 'U':
				case 'u':
					link->status = "unavailable";
					break;
				}
			}
		}
		link->groups = unrollGroups( link->groups );
		return;
	}
	if (link != NULL) {
		if (stricmp( word, "defaultGroup" ) == 0) {
			getWord( nextPart, word );
			if (*word != '\0') {
				link->defaultGroup = word;
			}
			return;
		}
		if (stricmp( word, "useAKA" ) == 0) {
			getWord( nextPart, word );
			if (*word != '\0') {
				link->useAKA = word;
			}
			return;
		}
		if (stricmp( word, "robotName" ) == 0) {
			getWord( nextPart, word );
			if (*word != '\0') {
				link->robotName = word;
			}
			return;
		}
		if (stricmp( word, "availAreas" ) == 0) {
			getWord( nextPart, word );
			if (*word != '\0') {
				link->availAreas = word;
			}
			return;
		}
		if (stricmp( word, "autocreated" ) == 0) {
			getWord( nextPart, word );
			if (*word != '\0') {
				link->autocreatedPath = word;
			}
			return;
		}
	}
}


void processLinks(const char* fileName) {
	std::cout << "converting links from " << fileName << std::endl;

	std::ifstream	in( fileName );
	if (in.fail()) {
		std::cout << "failed to open " << fileName << std::endl;
		return;
	}
	char	buffer[4096];
	char	word[4096];
	while (!in.eof()) {
		if (in.getline( buffer, 4096 )) {
			const char*	nextPart = getWord( buffer, word );
			if (stricmp( word, "include" ) == 0) {
				nextPart = getWord( nextPart, word );
				if (word[0] != '\0') {
					processLinks( word );
				}
			} else {
				processLinkString( buffer );
			}
		}
	}
	in.close();
}

void convertLinks() {
	link = NULL;

	processLinks( ADRENALIN_CFG );
	if (link != NULL) {
		links.push_back( link );
	}

	std::ofstream	out( "links.xml" );
	if (out.fail()) {
		std::cout << "Failed to open links.xml for writing\n";
		return;
	}

	out << "<links>\n";

	string	resolved;

	for (int i = 0; i < links.size(); i++) {

		link = links[i];

		if (link->priority > maxLinkPriority) {
			maxLinkPriority = link->priority;
		}

		out << "    <link>\n";
		out << "        <name>SysOp</name>\n";
		out << "        <address>" << link->address.c_str() << "</address>\n";
		if (link->useAKA.size() > 0) {
			out << "        <useAKA>" << link->useAKA << "</useAKA>\n";
		}
		out << "        <outbound type=\"binkley\" flavour=\"" << link->flavour.c_str() << "\" />\n";
		out << "        <security>\n";
		resolve( link->password.c_str(), resolved );
		out << "            <password>" << resolved.c_str() << "</password>\n";
		if (stricmp( link->routing.c_str(), "send" ) == 0) {
			link->sendPriority = link->priority;
		}
		if (stricmp( link->routing.c_str(), "receive" ) == 0) {
			link->receivePriority = link->priority;
		}
		if (stricmp( link->routing.c_str(), "bidirectional" ) == 0) {
			link->sendPriority = link->priority;
			link->receivePriority = link->priority;
		}
		out << "            <sendPriority>" << link->sendPriority << "</sendPriority>\n";
		out << "            <receivePriority>" << link->receivePriority << "</receivePriority>\n";
		// print groups
		out << "            <group>common</group>\n";
		char	group[2];
		group[1] = '\0';
		for (int j = 0; j < link->groups.size(); j++) {
			group[0] = link->groups[j];
			out << "            <group>" << group << "</group>\n";
		}
		out << "        </security>\n";
		if (link->allowAreaCreation) {
			out << "        <allowAreaCreation>\n";
			if (link->autocreatedPath.size() > 0) {
				out << "            <path>" << link->autocreatedPath << "</path>\n";
			}
			out << "            <defaultGroup>" << link->defaultGroup << "</defaultGroup>\n";
			out << "        </allowAreaCreation>\n";
		}
		if (link->status.size() > 0) {
			out << "        <status>" << link->status << "</status>\n";
		}
		if (link->robotName.size() > 0) {
			resolve( link->robotName.c_str(), resolved );
			out << "        <robotName>" << resolved.c_str() << "</robotName>\n";
		}
		if (link->availAreas.size() > 0) {
			out << "        <availAreas>" << link->availAreas << "</availAreas>\n";
		}
		out << "    </link>\n";
	}

	out << "</links>";

	out.close();	
}


void processAreaString(const char* s) {
	char		word[4096];
	const char*	nextPart = NULL;

	nextPart = getWord( s, word );
	if (stricmp( word, "area" ) == 0) {
		if (area != NULL) {
			areas.push_back( area );
		}
		area = new CArea();
		// name
		nextPart = getWord( nextPart, word );
		area->name = word;
		// path
		nextPart = getWord( nextPart, word );
		area->path = word;
		while (*nextPart != '\0') {
			bool	doubleQuoted = seeDoubleQuote( nextPart );
			nextPart = getWord( nextPart, word );
			if (*word == '\0') {
				break;
			}
			if (doubleQuoted) {
				area->desc = word;
			} else {
				switch (word[0]) {
				case 'S':
				case 's':
					area->routing = "send";
					break;
				case 'R':
				case 'r':
					area->routing = "receive";
					break;
				case 'B':
				case 'b':
					area->routing = "bidirectional";
					break;
				case 'P':
				case 'p':
					area->passthrough = true;
					break;
				case 'V':
				case 'v':
					area->vetoManualPurge = true;
					break;
				default:
					{
						const char*	slash = strstr( word, "/" );
						string	priority;
						if (slash == NULL) {
							priority = word;
						} else {
							priority.assign( word, slash - word );
							area->group = slash + 1;
						}
						int     scanResult = sscanf( priority.c_str(), "%d", &(area->priority) );
						if (scanResult == 0 || scanResult == EOF || area->priority < 0) {
							area->priority = 0;
						}
					}
				}
			}
		}
		return;
	}
	if (stricmp( word, "group" ) == 0) {
		// remember area
		if (area != NULL) {
			areas.push_back( area );
			area = NULL;
		}
		PAreaGroup	group = new CAreaGroup();
		// name
		nextPart = getWord( nextPart, word );
		group->name = word;
		// [desc]
		nextPart = getWord( nextPart, word );
		if (*word != '\0') {
			group->desc = word;
		}
		groups.push_back( group );
		return;
	}
	if (area != NULL) {
		if (stricmp( word, "openForResend" ) == 0) {
			area->openForResend = true;
			return;
		}
		if (stricmp( word, "linked" ) == 0) {
			while (*nextPart != '\0') {
				nextPart = getWord( nextPart, word );
				if (*word == '\0') {
					break;
				}
				const char*	addr = word+1;
				PLink	linked = new CLink();
				linked->routing = "";
				switch (word[0]) {
				case 'S':
				case 's':
					linked->routing = "send";
					break;
				case 'R':
				case 'r':
					linked->routing = "receive";
					break;
				case 'B':
				case 'b':
					linked->routing = "bidirectional";
					break;
				default:
					--addr;
				}
				linked->address = addr;
				area->linked.push_back( linked );
			}
			return;
		}
	}
}


void processAreas() {
	std::cout << "converting areas from " << AREAS_CFG << std::endl;

	std::ifstream	in( AREAS_CFG );
	if (in.fail()) {
		std::cout << "failed to open " << AREAS_CFG << std::endl;
		return;
	}
	char	buffer[4096];
	while (!in.eof()) {
		if (in.getline( buffer, 4096 )) {
			processAreaString( buffer );
		}
	}
	in.close();
}


void convertAreas() {
	area = NULL;
	processAreas();
	if (area != NULL) {
		areas.push_back( area );
	}

	// ensure that all groups are counted
	{
		for (int i = 0; i < areas.size(); i++) {
			string	groupName = areas[i]->group;
			int	j = 0;
			for (j = 0; j < groups.size(); j++) {
				if (stricmp( groups[j]->name.c_str(), groupName.c_str() ) == 0) {
					break;
				}
			}
			if (j >= groups.size()) {
				PAreaGroup	newGroup = new CAreaGroup();
				newGroup->name = groupName;
				groups.push_back( newGroup );
			}
		}
	}

	std::ofstream	out( "areas.xml" );
	if (out.fail()) {
		std::cout << "Failed to open areas.xml for writing\n";
		return;
	}

	string	resolved;

	out << "<areas>\n";
	for (int i = 0; i < groups.size(); i++) {
		PAreaGroup	group = groups[i];

		resolve( group->name.c_str(), resolved );
		out << "    <group name=\"" << resolved.c_str() << "\"";
		if (group->desc.size() > 0) {
			resolve( group->desc.c_str(), resolved );
			out << " description=\"" << resolved.c_str() << "\"";
		}
		out << ">\n";

		for (int j = 0; j < areas.size(); j++) {
			PArea	area = areas[j];
			if (stricmp( area->group.c_str(), group->name.c_str() ) == 0) {
				resolve( area->name.c_str(), resolved );
				out << "        <area name=\"" << resolved.c_str() << "\"\n";
				if (area->desc.size() > 0) {
					resolve( area->desc.c_str(), resolved );
					out << "              description=\"" << resolved.c_str() << "\"\n";
				}
				string	path = area->path;
				if (path.size() > 1) {
					if (path[path.size()-1] == '\\' && path[path.size()-2] != '\\') {
						path = path.substr( 0, path.size()-1 );
					}
				}
				out << "              path=\"" << path << "\"\n";

				int	sendPriority    = area->priority;
				int	receivePriority = area->priority;

				bool	send    = false;
				bool	receive = false;

				if (stricmp( area->routing.c_str(), "send" ) == 0) {
					// we can only send into area						
					if (receivePriority <= maxLinkPriority) {
						receivePriority = maxLinkPriority + 1;
					}
					send = true;
				}
				if (stricmp( area->routing.c_str(), "receive" ) == 0) {
					// we can only receive from area
					if (sendPriority <= maxLinkPriority) {
						sendPriority = maxLinkPriority + 1;
					}
					receive = true;
				}
				if (stricmp( area->routing.c_str(), "bidirectional" ) == 0) {
					send    = true;
					receive = true;
				}
				out << "              sendPriority=\"" << sendPriority << "\"\n";
				out << "              receivePriority=\"" << receivePriority << "\">\n";

				if (area->passthrough) {
					out << "            <passthrough />\n";
				}
				if (area->vetoManualPurge) {
					out << "            <vetoManualPurge />\n";
				}
				if (area->openForResend) {
					out << "            <openForResend />\n";
				}

				// print linked links
				int	linkedCount = area->linked.size();
				for (int k = 0; k < linkedCount; k++) {
					PLink	linked = area->linked[k];
					int		linkedSendPriority, linkedReceivePriority;
					getPriorities( linked->address, linkedSendPriority, linkedReceivePriority );
					out << "            <linked";
					if (linked->routing.size() > 0) {
						// let's check whether we need this circumstantiation
						bool	need = false;
						if (stricmp( linked->routing.c_str(), "send" ) == 0) {
							if (sendPriority > linkedSendPriority) {
								need = true;
							}
						}
						if (stricmp( linked->routing.c_str(), "receive" ) == 0) {
							if (receivePriority > linkedReceivePriority) {
								need = true;
							}
						}
						if (stricmp( linked->routing.c_str(), "bidirectional" ) == 0) {
							if (receivePriority > linkedReceivePriority ||
								sendPriority > linkedSendPriority)
							{
								need = true;
							}
						}
						if (need) {
							out << " routing=\"" << linked->routing.c_str() << "\"";
						}
					}
					else {
						// link is linked => in old version it was working (sending or receiving)
						if (send) {
							if (receive) {
								// bidirectional area
								if (sendPriority > linkedSendPriority || receivePriority > linkedReceivePriority) {
									// not enought priority
									out << " routing=\"bidirectional\"";
								}
							}
							else {
								// area to send
								if (sendPriority > linkedSendPriority || receivePriority <= linkedReceivePriority) {
									// not enought priority to send
									// or enought to receive
									// (should be only send)
									out << " routing=\"send\"";
								}
							}
						}
						else {
							if (receive) {
								// area to receive
								if (sendPriority <= linkedSendPriority || receivePriority > linkedReceivePriority) {
									// enought priority to send
									// or not enought to receive
									// (should be only receive)
									out << " routing=\"receive\"";
								}
							}
						}
					}
					out << ">" << linked->address << "</linked>\n";
				}

				out << "        </area>\n";
			}
		}
		out << "    </group>\n";
	}

	out << "</areas>\n";

	out.close();
}

bool backupCfg() {
	char	buffer[_MAX_PATH];

	for (int index = -1;;++index) {
		// check links
		if (index < 0) {
			strcpy( buffer, "links.xml" );
		} else {
			sprintf( buffer, "links.xml.%d.backup", index );
		}
		if (_access( buffer, 0 ) == 0 || errno != ENOENT) {
			continue;
		}
		// check areas
		if (index < 0) {
			strcpy( buffer, "areas.xml" );
		} else {
			sprintf( buffer, "areas.xml.%d.backup", index );
		}
		if (_access( buffer, 0 ) == 0 || errno != ENOENT) {
			continue;
		}
		break;
	}
	if (index < 0) {
		// nothing to backup, suppose that backup is ok :)
		return true;
	} else {
		// rename links
		sprintf( buffer, "links.xml.%d.backup", index );
		if (rename( "links.xml", buffer ) != 0 && errno != ENOENT) {
			return false;
		}
		// rename areas
		sprintf( buffer, "areas.xml.%d.backup", index );
		if (rename( "areas.xml", buffer ) != 0 && errno != ENOENT) {
			return false;
		}
		return true;
	}
}

void resolve(const char* data, string& resolvedData) {
	// reset resolved data
	resolvedData = "";

	string	entity;
	int		startPos = 0;
	int		endPos   = 0;
	while (data[endPos] != '\0') {
		bool	resolveTime = true;	// nice name? ;)
		switch (data[endPos]) {
		case '<':
			entity = "&lt;";
			break;
		case '>':
			entity = "&gt;";
			break;
		case '"':
			entity = "&quot;";
			break;
		case '\'':
			entity = "&apos;";
			break;
		case '&':
			entity = "&amp;";
			break;
		default:
			resolveTime = false;
		}
		if (resolveTime) {
			if (startPos < endPos) {
				resolvedData.append( data + startPos, endPos - startPos );
			}
			resolvedData += entity;
			++endPos;
			startPos = endPos;
		}
		else {
			++endPos;
		}
	}
	if (startPos < endPos) {
		resolvedData.append( data + startPos, endPos - startPos );
	}
}
